'use client'

import { useState } from 'react'
import { useRouter } from 'next/navigation'
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { ChevronLeft } from 'lucide-react'

export function BlockPage() {
  const router = useRouter()
  const [step, setStep] = useState(1)
  const [selectedType, setSelectedType] = useState('')
  const [selectedProperty, setSelectedProperty] = useState('')

  const inspectionTypes = [
    {
      id: 'entry',
      name: 'État des lieux d\'entrée',
      emoji: '🔑',
      description: 'Documentation détaillée de l\'état du bien avant l\'emménagement du locataire'
    },
    {
      id: 'exit',
      name: 'État des lieux de sortie',
      emoji: '🚪',
      description: 'Évaluation complète de l\'état du bien au départ du locataire'
    },
    {
      id: 'routine',
      name: 'Inspection de routine',
      emoji: '📋',
      description: 'Vérification périodique de l\'état général du bien'
    },
    {
      id: 'damage',
      name: 'Évaluation des dommages',
      emoji: '🔍',
      description: 'Inspection suite à un incident ou dégât signalé'
    },
    {
      id: 'purchase',
      name: 'Inspection avant achat',
      emoji: '🏠',
      description: 'Diagnostics immobiliers complets avant l\'achat d\'un bien'
    },
    {
      id: 'compliance',
      name: 'Inspection de conformité',
      emoji: '✅',
      description: 'Vérification de la conformité du bien aux normes et réglementations'
    },
    {
      id: 'prework',
      name: 'Inspection préalable aux travaux',
      emoji: '🔨',
      description: 'Constat d\'huissier avant le début de travaux importants'
    },
    {
      id: 'energy',
      name: 'Audit énergétique',
      emoji: '🌡️',
      description: 'Évaluation de la performance énergétique du bien'
    }
  ]

  const properties = [
    { id: '1', name: 'Appartement Paris', type: 'Appartement' },
    { id: '2', name: 'Maison Bordeaux', type: 'Maison' },
    { id: '3', name: 'Studio Lyon', type: 'Studio' }
  ]

  const handleSelectType = (typeId: string) => {
    setSelectedType(typeId)
    setStep(2)
  }

  const handleSelectProperty = (propertyId: string) => {
    setSelectedProperty(propertyId)
    setStep(3)
  }

  return (
    <div className="min-h-screen bg-gray-50 p-4">
      <div className="max-w-2xl mx-auto">
        <div className="flex items-center justify-between mb-6">
          <button 
            onClick={() => step > 1 ? setStep(step - 1) : router.back()}
            className="p-2 hover:bg-gray-100 rounded-full"
          >
            <ChevronLeft className="h-6 w-6" />
          </button>
          <h1 className="text-2xl font-bold text-center">Nouvelle inspection</h1>
          <div className="w-10" /> {/* Spacer for centering */}
        </div>

        <div className="flex justify-center mb-8">
          {[1, 2, 3].map((i) => (
            <div
              key={i}
              className={`w-16 h-2 rounded-full mx-1 ${
                step >= i ? 'bg-[#8B5CF6]' : 'bg-gray-200'
              }`}
            />
          ))}
        </div>

        {step === 1 && (
          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-[#8B5CF6] mb-6">Type d'inspection</h2>
            <div className="grid grid-cols-1 gap-4">
              {inspectionTypes.map((type) => (
                <Card 
                  key={type.id}
                  className={`cursor-pointer hover:border-[#8B5CF6] transition-colors ${
                    selectedType === type.id ? 'border-[#8B5CF6] bg-[#8B5CF6]/5' : ''
                  }`}
                  onClick={() => handleSelectType(type.id)}
                >
                  <CardContent className="p-4">
                    <div className="flex items-start space-x-4">
                      <span className="text-2xl">{type.emoji}</span>
                      <div>
                        <h3 className="font-semibold text-lg">{type.name}</h3>
                        <p className="text-gray-500 text-sm">{type.description}</p>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        )}

        {step === 2 && (
          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-[#8B5CF6] mb-6">Sélectionnez une propriété</h2>
            <div className="grid grid-cols-1 gap-4">
              {properties.map((property) => (
                <Card 
                  key={property.id}
                  className={`cursor-pointer hover:border-[#8B5CF6] transition-colors ${
                    selectedProperty === property.id ? 'border-[#8B5CF6] bg-[#8B5CF6]/5' : ''
                  }`}
                  onClick={() => handleSelectProperty(property.id)}
                >
                  <CardContent className="p-4">
                    <div className="flex items-start space-x-4">
                      <span className="text-2xl">🏠</span>
                      <div>
                        <h3 className="font-semibold text-lg">{property.name}</h3>
                        <p className="text-gray-500 text-sm">{property.type}</p>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        )}

        {step === 3 && (
          <div className="space-y-4">
            <h2 className="text-2xl font-bold text-[#8B5CF6] mb-6">Planifier l'inspection</h2>
            <Card>
              <CardContent className="p-6 space-y-6">
                <div className="flex items-center space-x-4">
                  <span className="text-2xl">
                    {inspectionTypes.find(t => t.id === selectedType)?.emoji}
                  </span>
                  <div>
                    <h3 className="font-semibold text-lg">
                      {inspectionTypes.find(t => t.id === selectedType)?.name}
                    </h3>
                    <p className="text-gray-500">
                      {properties.find(p => p.id === selectedProperty)?.name}
                    </p>
                  </div>
                </div>
                <div className="space-y-4">
                  <input
                    type="date"
                    className="w-full p-2 border rounded-lg"
                    min={new Date().toISOString().split('T')[0]}
                  />
                  <input
                    type="time"
                    className="w-full p-2 border rounded-lg"
                  />
                  <Button 
                    className="w-full bg-[#8B5CF6] hover:bg-[#7C3AED] text-white h-12"
                    onClick={() => router.push('/inspections/confirmation')}
                  >
                    Confirmer l'inspection
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        )}
      </div>
    </div>
  )
}