'use client'

import { useState } from 'react'
import { useRouter } from 'next/navigation'
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/components/ui/tabs"
import { Bell, ChevronLeft, CheckCircle, AlertTriangle, Calendar, Trash2 } from 'lucide-react'

type Notification = {
  id: string;
  type: 'inspection' | 'property' | 'system';
  title: string;
  message: string;
  date: string;
  read: boolean;
}

export function BlockPage() {
  const router = useRouter()
  const [notifications, setNotifications] = useState<Notification[]>([
    { id: '1', type: 'inspection', title: 'Inspection planifiée', message: 'Rappel : Inspection de l\'Appartement Paris demain à 14h', date: '2024-06-14', read: false },
    { id: '2', type: 'property', title: 'Mise à jour de propriété', message: 'La Maison Bordeaux a été mise à jour avec de nouvelles photos', date: '2024-06-13', read: true },
    { id: '3', type: 'system', title: 'Maintenance du système', message: 'CasaOK sera en maintenance le 20 juin de 2h à 4h du matin', date: '2024-06-12', read: false },
    { id: '4', type: 'inspection', title: 'Rapport disponible', message: 'Le rapport d\'inspection pour le Studio Lyon est maintenant disponible', date: '2024-06-11', read: false },
    { id: '5', type: 'property', title: 'Nouvelle propriété ajoutée', message: 'La propriété "Villa Cannes" a été ajoutée à votre portfolio', date: '2024-06-10', read: true },
  ])

  const markAsRead = (id: string) => {
    setNotifications(notifications.map(notif => 
      notif.id === id ? { ...notif, read: true } : notif
    ))
  }

  const deleteNotification = (id: string) => {
    setNotifications(notifications.filter(notif => notif.id !== id))
  }

  const getIcon = (type: string) => {
    switch(type) {
      case 'inspection': return <Calendar className="h-5 w-5 text-blue-500" />;
      case 'property': return <CheckCircle className="h-5 w-5 text-green-500" />;
      case 'system': return <AlertTriangle className="h-5 w-5 text-yellow-500" />;
      default: return <Bell className="h-5 w-5 text-gray-500" />;
    }
  }

  return (
    <div className="min-h-screen bg-gray-50 p-4">
      <div className="max-w-2xl mx-auto">
        <div className="flex items-center justify-between mb-6">
          <button 
            onClick={() => router.back()}
            className="p-2 hover:bg-gray-100 rounded-full"
          >
            <ChevronLeft className="h-6 w-6" />
          </button>
          <h1 className="text-2xl font-bold text-center">Notifications</h1>
          <div className="w-10" />
        </div>

        <Tabs defaultValue="all" className="w-full">
          <TabsList className="grid w-full grid-cols-4 mb-4">
            <TabsTrigger value="all">Toutes</TabsTrigger>
            <TabsTrigger value="inspection">Inspections</TabsTrigger>
            <TabsTrigger value="property">Propriétés</TabsTrigger>
            <TabsTrigger value="system">Système</TabsTrigger>
          </TabsList>
          {['all', 'inspection', 'property', 'system'].map((tab) => (
            <TabsContent key={tab} value={tab}>
              {notifications
                .filter(notif => tab === 'all' || notif.type === tab)
                .map((notif) => (
                  <Card key={notif.id} className={`mb-4 ${notif.read ? 'bg-white' : 'bg-blue-50'}`}>
                    <CardContent className="p-4">
                      <div className="flex items-start justify-between">
                        <div className="flex items-start space-x-3">
                          {getIcon(notif.type)}
                          <div>
                            <h3 className="font-semibold text-lg">{notif.title}</h3>
                            <p className="text-gray-600">{notif.message}</p>
                            <p className="text-sm text-gray-400 mt-1">{notif.date}</p>
                          </div>
                        </div>
                        <div className="flex space-x-2">
                          {!notif.read && (
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => markAsRead(notif.id)}
                            >
                              <CheckCircle className="h-4 w-4" />
                            </Button>
                          )}
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => deleteNotification(notif.id)}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
            </TabsContent>
          ))}
        </Tabs>
      </div>
    </div>
  )
}