'use client'

import { useState } from 'react'
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Label } from "@/components/ui/label"
import { Switch } from "@/components/ui/switch"
import { Separator } from "@/components/ui/separator"
import { ChevronLeft, Lock, Fingerprint, Smartphone, Eye, KeyRound, LogOut } from 'lucide-react'
import Link from 'next/link'

export function BlockPage() {
  const [securitySettings, setSecuritySettings] = useState({
    twoFactor: false,
    biometricLogin: true,
    passwordlessLogin: false,
    loginNotifications: true,
  })

  const handleToggle = (key: keyof typeof securitySettings) => {
    setSecuritySettings(prev => ({ ...prev, [key]: !prev[key] }))
  }

  const securityOptions = [
    { 
      id: 'twoFactor', 
      name: 'Authentification à deux facteurs', 
      description: 'Ajouter une couche de sécurité supplémentaire à votre compte',
      icon: Smartphone
    },
    { 
      id: 'biometricLogin', 
      name: 'Connexion biométrique', 
      description: 'Utiliser votre empreinte digitale ou la reconnaissance faciale pour vous connecter',
      icon: Fingerprint
    },
    { 
      id: 'passwordlessLogin', 
      name: 'Connexion sans mot de passe', 
      description: 'Se connecter avec un lien envoyé par e-mail',
      icon: Lock
    },
    { 
      id: 'loginNotifications', 
      name: 'Notifications de connexion', 
      description: 'Recevoir une notification lors d\'une nouvelle connexion à votre compte',
      icon: Eye
    },
  ]

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="sticky top-0 z-10 bg-white px-4 py-3 shadow-sm">
        <div className="flex items-center justify-between max-w-md mx-auto">
          <Link href="/settings">
            <Button variant="ghost" size="icon" className="text-gray-600 hover:text-gray-900">
              <ChevronLeft className="h-5 w-5" />
              <span className="sr-only">Retour</span>
            </Button>
          </Link>
          <h1 className="text-lg font-semibold text-gray-900">Sécurité</h1>
          <div className="w-8"></div>
        </div>
      </div>

      <div className="p-4 max-w-md mx-auto space-y-4">
        <Card className="overflow-hidden">
          <CardContent className="p-0">
            {securityOptions.map((option, index) => (
              <div key={option.id}>
                {index > 0 && <Separator className="my-0" />}
                <div className="flex items-center p-4 space-x-3">
                  <div className="flex-shrink-0 w-8 h-8 bg-purple-100 rounded-full flex items-center justify-center">
                    <option.icon className="h-4 w-4 text-purple-600" />
                  </div>
                  <div className="flex-grow min-w-0">
                    <Label htmlFor={option.id} className="flex flex-col">
                      <span className="font-medium text-sm text-gray-900 truncate">{option.name}</span>
                      <span className="text-xs text-gray-500 truncate">{option.description}</span>
                    </Label>
                  </div>
                  <Switch
                    id={option.id}
                    checked={securitySettings[option.id as keyof typeof securitySettings]}
                    onCheckedChange={() => handleToggle(option.id as keyof typeof securitySettings)}
                    className="data-[state=checked]:bg-purple-600"
                  />
                </div>
              </div>
            ))}
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <Button className="w-full bg-purple-600 hover:bg-purple-700 text-white flex items-center justify-center space-x-2">
              <KeyRound className="h-4 w-4" />
              <span>Changer le mot de passe</span>
            </Button>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <Button variant="outline" className="w-full text-red-600 border-red-200 hover:bg-red-50 flex items-center justify-center space-x-2">
              <LogOut className="h-4 w-4" />
              <span>Déconnecter toutes les sessions</span>
            </Button>
          </CardContent>
        </Card>
      </div>
    </div>
  )
}